<?php
/* --------------------------------------------------------------
  OnGetSellingUnitModelEventListener.php 2020-06-24
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2020 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Shop\Attributes\SellingUnitModel\Database\Listener;

use Gambio\Shop\Attributes\SellingUnitModel\Database\Exceptions\OptionValueIdDoesNotExistsException;
use Gambio\Shop\Attributes\SellingUnitModel\Database\Repository\DTO\AttributesModelDto;
use Gambio\Shop\Attributes\SellingUnitModel\Database\Repository\DTO\OptionValuesIdDto;
use Gambio\Shop\Attributes\SellingUnitModel\Database\Service\ReadServiceInterface;
use Gambio\Shop\ProductModifiers\Modifiers\ValueObjects\ModifierIdentifierInterface;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationId;
use Gambio\Shop\Properties\SellingUnitModel\Database\Exceptions\PropertiesCombinationIdDoesNotExistException;
use Gambio\Shop\Properties\SellingUnitModel\Database\Repository\DTO\PropertiesCombinationIdDto;
use Gambio\Shop\Properties\SellingUnitModel\Database\Repository\DTO\CombinationModelDto;
use Gambio\Shop\Properties\SellingUnitModel\Database\Service\ReadServiceInterface as PropertiesReadServiceInterface;
use Gambio\Shop\SellingUnit\Database\Configurations\AppendPropertiesModelConfiguration;
use Gambio\Shop\SellingUnit\Database\Unit\Events\OnGetSellingUnitModelEvent;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Model;

/**
 * Class OnGetSellingUnitModelEventListener
 * @package Gambio\Shop\Attributes\SellingUnitModel\Database\Listener
 */
class OnGetSellingUnitModelEventListener
{
    /**
     * @var ReadServiceInterface
     */
    protected $service;

    /**
     * @var PropertiesReadServiceInterface
     */
    protected $propertiesService;

    /**
     * @var AppendPropertiesModelConfiguration
     */
    protected $appendPropertiesModel;


    /**
     * OnGetSellingUnitModelEventListener constructor.
     *
     * @param ReadServiceInterface               $service
     * @param PropertiesReadServiceInterface     $propertiesService
     * @param AppendPropertiesModelConfiguration $appendPropertiesModel
     */
    public function __construct(
        ReadServiceInterface $service,
        PropertiesReadServiceInterface $propertiesService,
        AppendPropertiesModelConfiguration $appendPropertiesModel
    ) {
        $this->service               = $service;
        $this->propertiesService     = $propertiesService;
        $this->appendPropertiesModel = $appendPropertiesModel;
    }


    /**
     * @param OnGetSellingUnitModelEvent $event
     */
    public function __invoke(OnGetSellingUnitModelEvent $event)
    {
        $modifiers = $event->modifierIdentifiers();

        if (count($modifiers)) {

            $model = $event->model()->value();

            // skip attribute model, if product has a property model and appending of model numbers is inactive
            if ($this->appendPropertiesModel->value() === false && $event->combinationId() !== null) {
                $combinationModelDto = $this->getCombinationModel($event->combinationId());

                if ($combinationModelDto->combinationModel() !== '') {
                    return;
                }
            }

            foreach ($modifiers as $modifier) {

                if ($modifier->type() === 'attribute') {
                    $modelDto = $this->modelDto($modifier);

                    if ($modelDto->model() === '') {
                        continue;
                    }

                    if ($model !== '') {
                        $model .= '-';
                    }

                    $model .= $modelDto->model();
                }
            }

            $event->setModel(new Model($model));
        }
    }


    /**
     * @param ModifierIdentifierInterface $modifier
     *
     * @return AttributesModelDto
     */
    protected function modelDto(ModifierIdentifierInterface $modifier): AttributesModelDto
    {
        $dto = new OptionValuesIdDto((int)$modifier->value());
        try {
            $result = $this->service->attributeModelByOptionValuesId($dto);
        } catch (OptionValueIdDoesNotExistsException $exception) {
            unset($exception);
            // if the model is empty it gets skipped
            $result = new AttributesModelDto('');
        }

        return $result;
    }


    /**
     * @param CombinationId $combinationId
     *
     * @return CombinationModelDto
     * @throws PropertiesCombinationIdDoesNotExistException
     */
    protected function getCombinationModel(CombinationId $combinationId): CombinationModelDto
    {
        $dto = new PropertiesCombinationIdDto($combinationId->value());

        return $this->propertiesService->combinationModel($dto);
    }
}